definition module StdWindow


//	********************************************************************************
//	Clean Standard Object I/O library, version 1.0.1
//	
//	StdWindow defines functions on windows and dialogues.
//	********************************************************************************


import	StdMaybe, StdWindowDef
from	StdPSt			import PSt, IOSt
from	StdControlClass	import Controls, ControlState


//	Functions applied to non-existent windows or unknown ids have no effect.
class Windows wdef
where
	openWindow		:: .ls !(wdef .ls (PSt .l .p)) !(PSt .l .p)
								   -> (!ErrorReport,!PSt .l .p)
	getWindowType	::      (wdef .ls .ps)	-> WindowType

class Dialogs wdef
where
	openDialog		:: .ls !(wdef .ls (PSt .l .p)) !(PSt .l .p)
								   -> (!ErrorReport,!PSt .l .p)
	openModalDialog	:: .ls !(wdef .ls (PSt .l .p)) !(PSt .l .p)
								   -> (!ErrorReport,!PSt .l .p)
	getDialogType	::      (wdef .ls .ps)	-> WindowType

/*	open(Window/Dialog) opens the given window(dialog).
	If the Window(Dialog) has no WindowIndex attribute (see StdWindowDef), then the 
	new window is opened frontmost. 
	If the Window(Dialog) has a WindowIndex attribute, then the new window is 
	opened behind the window indicated by the integer index:
		Index value 1 indicates the top-most window. 
		Index value M indicates the bottom-most modal window, if there are M modal 
			windows. 
		Index value N indicates the bottom-most window, if there are N windows. 
		If index<M, then the new window is added behind the bottom-most modal window
			(at index M). 
		If index>N, then the new window is added behind the bottom-most window
			(at index N). 
	openModalDialog always opens a window at the front-most position. 
	openWindow may not be permitted to open a window depending on its 
		DocumentInterface (see the comments at the ShareProcesses instances in 
		module StdProcess).
	In case the window does not have an Id, it will obtain an Id which is fresh with 
	respect to the current set of windows. The Id can be reused after closing this 
	window. 
	In case a window with the same Id is already open the window will not be opened. 
	In case controls are opened with duplicate Ids, the window will not be opened.
	openModalDialog terminates when:
		the window has been closed (by means of closeWindow), or the process has 
		been terminated (by means of closeProcess).
*/

instance Windows (Window c)	| Controls c
instance Dialogs (Dialog c) | Controls c


closeWindow :: !Id !(PSt .l .p) -> PSt .l .p
/*	If the indicated window is not an inactive modal window, then closeWindow closes 
	the window.
	In case the Id of the window was generated by open(Window/Dialog), it will 
	become reusable for new windows/dialogues.
	In case of unknown Id, closeWindow does nothing.
*/

closeControls :: !Id [Id] !Bool	!(IOSt .l .p) -> IOSt .l .p
/*	closeControls removes the indicated controls (second argument) from the 
	indicated window (first argument) and recalculates the layout iff the Boolean 
	argument is True.
*/


openControls		:: !Id     .ls (cdef .ls (PSt .l .p)) !(IOSt .l .p)
										  -> (!ErrorReport,!IOSt .l .p)
										  |  Controls cdef
openCompoundControls:: !Id !Id .ls (cdef .ls (PSt .l .p)) !(IOSt .l .p)
										  -> (!ErrorReport,!IOSt .l .p)
										  |  Controls cdef
/*	openControls
		adds the given controls argument to the indicated window. 
	openCompoundControls
		adds the given controls argument to the indicated compound control.
	Both functions have no effect in case the indicated window/dialog/compound 
	control could not be found (ErrorUnknownObject) or if controls are opened with 
	duplicate Ids (ErrorIdsInUse).
*/


setControlPos :: !Id !Id !ItemPos	!(IOSt .l .p) -> (!Bool,!IOSt .l .p)
/*	setControlPos changes the current layout position of the indicated control to 
	the new position. 
	If there are relatively laynout controls, then their layout also changes. The 
	window is not resized.
	The Boolean result is False iff the window or control id are unknown, or if the
	new ItemPos refers to an unknown control. 
*/


controlSize :: !(cdef .ls (PSt .l .p))
					 !(Maybe (Int,Int)) !(Maybe (Int,Int)) !(Maybe (Int,Int))
					 !(IOSt .l .p)
			-> (!Size,!IOSt .l .p)
			|  Controls cdef
/*	controlSize calculates the size of the given control definition as it would be 
	opened as an element of a window/dialog. 
	The Maybe arguments are the prefered horizontal margins, vertical margins, and 
	item spaces (see also the (Window/Control)(H/V)Margin and 
	(Window/Control)ItemSpace attributes). If Nothing is specified, their default 
	values are used.
*/


hideWindows		:: ![Id]	!(IOSt .l .p) -> IOSt .l .p
showWindows		:: ![Id]	!(IOSt .l .p) -> IOSt .l .p
getHiddenWindows::			!(IOSt .l .p) -> (![Id],!IOSt .l .p)
getShownWindows	::			!(IOSt .l .p) -> (![Id],!IOSt .l .p)
/*	(hide/show)Windows hides/shows the indicated modeless windows (modal dialogues 
	can not be hidden). 
	get(Hidden/Shown)Windows yields the list of currently visible/invisible windows.
*/


activateWindow	:: !Id		!(IOSt .l .p) -> IOSt .l .p
/*	activateWindow makes the indicated window the active window. 
	If the window was hidden, then it will become shown. 
	If there are modal dialogues, then the window will be placed behind the last 
	modal dialog.
	activateWindow has no effect in case the window is unknown or is a modal dialog.
*/

getActiveWindow	::			!(IOSt .l .p) -> (!Maybe Id,!IOSt .l .p)
/*	getActiveWindow returns the Id of the window that currently has the input focus
	of the interactive process. 
	Nothing is returned if there is no such window.
	Note that hidden windows never are active windows, and that modal windows never
	are hidden.
*/


stackWindow		:: !Id !Id	!(IOSt .l .p) -> IOSt .l .p
/*	stackWindow id1 id2 places the window with id1 behind the window with id2. 
	If id1 or id2 is unknown, or id1 indicates a modal window, stackWindow does 
	nothing.
	If id2 indicates a modal window, then the window with id1 is placed behind the 
	last modal window. 
*/

getWindowStack	:: !(IOSt .l .p) -> (![(Id,WindowType)],!IOSt .l .p)
getWindowsStack	:: !(IOSt .l .p) -> (![Id],				!IOSt .l .p)
getDialogsStack	:: !(IOSt .l .p) -> (![Id],				!IOSt .l .p)
/*	getWindowStack returns the Ids and WindowTypes of all currently open windows 
	(including the hidden windows), in the current stacking order starting with the 
	active window.
	get(Windows/Dialogs)Stack is equal to getWindowStack, restricted to Windows 
	instances and Dialogs instances respectively.
*/


getDefaultHMargin	::		!(IOSt .l .p) -> ((Int,Int),		!IOSt .l .p)
getDefaultVMargin	::		!(IOSt .l .p) -> ((Int,Int),		!IOSt .l .p)
getDefaultItemSpace	::		!(IOSt .l .p) -> ((Int,Int),		!IOSt .l .p)
getWindowHMargin	:: !Id	!(IOSt .l .p) -> (!Maybe (Int,Int),	!IOSt .l .p)
getWindowVMargin	:: !Id	!(IOSt .l .p) -> (!Maybe (Int,Int),	!IOSt .l .p)
getWindowItemSpace	:: !Id	!(IOSt .l .p) -> (!Maybe (Int,Int),	!IOSt .l .p)
/*	getDefault((H/V)Margin)/ItemSpace) return the default values for the horizontal
	and vertical window/dialogue margins and item spaces.
	getWindow((H/V)Margin/ItemSpace) return the current horizontal and vertical 
	margins and item spaces of the indicated window. These will have the default 
	values in case they are not specified.
	In case the window does not exist, Nothing is yielded.
*/


enableWindow			:: !Id	!(IOSt .l .p) -> IOSt .l .p
disableWindow			:: !Id	!(IOSt .l .p) -> IOSt .l .p
enableWindowMouse		:: !Id	!(IOSt .l .p) -> IOSt .l .p
disableWindowMouse		:: !Id	!(IOSt .l .p) -> IOSt .l .p
enableWindowKeyboard	:: !Id	!(IOSt .l .p) -> IOSt .l .p
disableWindowKeyboard	:: !Id	!(IOSt .l .p) -> IOSt .l .p
/*	(en/dis)ableWindow
		(en/dis)ables the indicated window.
	(en/dis)ableWindowMouse
		(en/dis)ables mouse handling of the indicated window.
	(en/dis)ableWindowKeyboard
		(en/dis)ables keyboard handling of the indicated window.
	Disabling a window overrules the SelectStates of its elements, which all become 
	Unable. 
	Reenabling the window reestablishes the SelectStates of its elements.
	The functions have no effect in case of invalid Ids or Dialogs instances.
	The latter four functions also have no effect in case the Window does not have 
	the indicated attribute.
*/


getWindowSelectState		:: !Id	!(IOSt .l .p) ->(!Maybe SelectState,!IOSt .l .p)
getWindowMouseSelectState	:: !Id	!(IOSt .l .p) ->(!Maybe SelectState,!IOSt .l .p)
getWindowKeyboardSelectState:: !Id	!(IOSt .l .p) ->(!Maybe SelectState,!IOSt .l .p)
/*	getWindowSelectState
		yields the current SelectState of the indicated window.
	getWindow(Mouse/Keyboard)SelectState
		yields the current SelectState of the mouse/keyboard of the indicated 
		window.
	The functions return Nothing in case of invalid Ids or Dialogs instances or if 
	the Window does not have the indicated attribute.
*/


getWindowMouseStateFilter	:: !Id							!(IOSt .l .p)
							-> (!Maybe MouseStateFilter,	! IOSt .l .p)
getWindowKeyboardStateFilter:: !Id							!(IOSt .l .p)
							-> (!Maybe KeyboardStateFilter,	! IOSt .l .p)
setWindowMouseStateFilter	:: !Id !MouseStateFilter		!(IOSt .l .p)
														   -> IOSt .l .p
setWindowKeyboardStateFilter:: !Id !KeyboardStateFilter		!(IOSt .l .p)
														   -> IOSt .l .p
/*	getWindow(Mouse/Keyboard)StateFilter yields the current 
		(Mouse/Keyboard)StateFilter of the indicated window. Nothing is yielded in 
		case the window does not exist or has no Window(Mouse/Keyboard) attribute.
	setWindow(Mouse/Keyboard)StateFilter replaces the current 
		(Mouse/Keyboard)StateFilter of the indicated window. If the indicated window
		does not exist the function has no effect.
*/


drawInWindow	:: !Id ![DrawFunction]		!(IOSt .l .p) -> IOSt .l .p
/*	drawInWindow applies the list of drawing functions in left-to-right order to the
	picture of the indicated window (behind all controls). 
	drawInWindow has no effect in case the window is unknown or is a Dialog.
*/

updateWindow	:: !Id !(Maybe ViewFrame)	!(IOSt .l .p) -> IOSt .l .p
/*	updateWindow applies the WindowLook attribute function of the indicated window.
	The SelectState argument of the Look attribute is the current SelectState of the
	window.
	The UpdateState argument of the Look attribute is 
		{oldFrame=frame,newFrame=frame,updArea=[frame]}
	where frame depends on the optional ViewFrame argument:
		in case of (Just rectangle):
			the intersection of the current ViewFrame of the window and rectangle.
		in case of Nothing:
			the current ViewFrame of the window. 
	updateWindow has no effect in case of unknown windows, or if the indicated 
	window is a Dialog, or the window has no WindowLook attribute, or the optional 
	viewframe argument is empty.
*/

setWindowLook	:: !Id !Bool !Look	!(IOSt .l .p) -> IOSt .l .p
getWindowLook	:: !Id				!(IOSt .l .p) -> (!Maybe Look,!IOSt .l .p)
/*	setWindowLook sets the Look of the indicated window. 
		The window is redrawn only if the Boolean argument is True. 
		setWindowLook has no effect in case the window does not exist, or is a 
		Dialog.
	getWindowLook returns the (Just Look) of the indicated window. 
		In case the window does not exist, or is a Dialog, or has no WindowLook 
		attribute, the result is Nothing. 
*/


setWindowPos	:: !Id !ItemPos	!(IOSt .l .p) -> IOSt .l .p
getWindowPos	:: !Id			!(IOSt .l .p) -> (!Maybe ItemOffset,!IOSt .l .p)
/*	setWindowPos places the window at the indicated position. 
		If the ItemPos argument refers to the Id of an unknown window (in case of 
		LeftOf/RightTo/Above/Below), setWindowPos has no effect. 
		If the ItemPos argument is one of (LeftOf/RightTo/Above/Below)Prev, then the
		previous window is the window that is before the window in the current 
		stacking order. 
		If the window is frontmost, setWindowPos has no effect. setWindowPos also 
		has no effect if the window would be moved outside the screen, or if the Id
		is unknown or refers to a modal Dialog.
	getWindowPos returns the current item offset position of the indicated window.
		The corresponding ItemPos is (LeftTop,offset). Nothing is returned in case 
		the window does not exist.
*/


moveWindowViewFrame	:: !Id Vector !(IOSt .l .p) -> IOSt .l .p
/*	moveWindowViewFrame moves the orientation of the view frame of the indicated 
	window over the given vector, and updates the window if necessary. The view 
	frame is not moved outside the ViewDomain of the window. 
	In case of unknown Id, or of Dialogs, moveWindowViewFrame has no effect.
*/

getWindowViewFrame	:: !Id !(IOSt .l .p) -> (!ViewFrame,!IOSt .l .p)
/*	getWindowViewFrame returns the current view frame of the window in terms of the
	ViewDomain. Note that in case of a Dialog, getWindowViewFrame returns 
	{zero,size}.
	In case of unknown windows, the ViewFrame result is zero.
*/

setWindowViewSize	:: !Id Size	!(IOSt .l .p) -> IOSt .l .p
getWindowViewSize	:: !Id		!(IOSt .l .p) -> (!Size,!IOSt .l .p)
/*	setWindowViewSize
		sets the size of the view frame of the indicated window as given, and 
		updates the window if necessary. The size is fit between the minimum size 
		and the screen dimensions. 
		In case of unknown Ids, or of Dialogs, setWindowViewSize has no effect.
	getWindowViewSize yields the current size of the view frame of the indicated 
		window. If the window does not exist, zero is returned.
*/

setWindowViewDomain	:: !Id ViewDomain	!(IOSt .l .p) -> IOSt .l .p
getWindowViewDomain	:: !Id				!(IOSt .l .p)
				   -> (!Maybe ViewDomain,!IOSt .l .p)
/*	setWindowViewDomain
		sets the view domain of the indicated window as given. The window view frame
		is moved such that a maximum portion of the view domain is visible. The 
		window is not resized.
		In case of unknown Ids, or of Dialogs, setWindowViewDomain has no effect.
	getWindowViewDomain
		returns the current ViewDomain of the indicated window. 
		Nothing is returned in case the window does not exist or is a Dialog.
*/


setWindowTitle	:: !Id Title       !(IOSt .l .p) -> IOSt .l .p
setWindowOk		:: !Id Id          !(IOSt .l .p) -> IOSt .l .p
setWindowCancel	:: !Id Id          !(IOSt .l .p) -> IOSt .l .p
setWindowCursor	:: !Id CursorShape !(IOSt .l .p) -> IOSt .l .p
getWindowTitle	:: !Id             !(IOSt .l .p) -> (!Maybe Title,      !IOSt .l .p)
getWindowOk		:: !Id             !(IOSt .l .p) -> (!Maybe Id,         !IOSt .l .p)
getWindowCancel	:: !Id             !(IOSt .l .p) -> (!Maybe Id,         !IOSt .l .p)
getWindowCursor	:: !Id             !(IOSt .l .p) -> (!Maybe CursorShape,!IOSt .l .p)
/*	setWindow(Title/Ok/Cancel/Cursor) set the indicated window attributes.
		In case of unknown Ids, these functions have no effect.
	getWindow(Title/Ok/Cancel/Cursor) get the indicated window attributes.
		In case of unknown Ids, the result is Nothing.
*/
